/*
 * idle_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * enet_io idle task monitors and manages changes to IP address.
 *
 * DisplayIP is used to output the IP address in a human-readable format.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 * Open a terminal with 115,200 8-N-1 to see the output for this demo.
 *
 */

/* Standard includes. */
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"
#include "semphr.h"

/* Hardware includes. */
#include "utils/lwiplib.h"
#include "utils/ustdlib.h"
#include "utils/uartstdio.h"
#include "lwip/stats.h"
/*-----------------------------------------------------------*/

/*
 * The current IP address.
 */
static uint32_t g_ui32IPAddress;

/*
 * DisplayIP is used to output the IP address in a human-readable format.
 */
static void DisplayIP(uint32_t ui32IP);
/*-----------------------------------------------------------*/

void
vApplicationIdleHook(void)
{
uint32_t ui32Temp;

    /* Get the current IP address. */
    ui32Temp = lwIPLocalIPAddrGet();

    /* See if the IP address has changed. */
    if(ui32Temp != g_ui32IPAddress)
    {
        /* Save the current IP address. */
        g_ui32IPAddress = ui32Temp;

        /* Update the display of the IP address. */
        DisplayIP(ui32Temp);
    }
}
/*-----------------------------------------------------------*/

static void
DisplayIP(uint32_t ui32IP)
{
char pcBuf[16];

    /* Has IP address been acquired? */
    if(ui32IP == 0)
    {
        /* If there is no IP address, indicate that one is being acquired. */
        UARTprintf("Acquiring IP Address...\n");
        return;
    }
    else if(ui32IP == 0xffffffff)
    {
        /* If link is inactive, indicate that. */
        UARTprintf("Waiting for link.\n");
        return;
    }

    /* Convert the IP Address into a string. */
    usprintf(pcBuf, "%d.%d.%d.%d", ui32IP & 0xff, (ui32IP >> 8) & 0xff,
             (ui32IP >> 16) & 0xff, (ui32IP >> 24) & 0xff);

    /* Display the string on Debug UART. */
    UARTprintf("IP Address: ");
    UARTprintf(pcBuf);
    UARTprintf("\nOpen a browser and enter the IP address to access the "
               "web server.\n");
}
/*-----------------------------------------------------------*/
